<?php

namespace Elementor;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Group_Control_Background;
use Elementor\Group_Control_Text_Stroke;

/**
 * Elementor number box widget.
 *
 * Elementor widget that displays an icon, a headline and a text.
 *
 * @since 1.0.0
 */
class OSF_Widget_Number_Box extends Widget_Base {

    /**
     * Get widget name.
     *
     * Retrieve icon box widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'number-box';
    }

    /**
     * Get widget title.
     *
     * Retrieve number box widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title() {
        return __('Number Box', 'makbet');
    }

    /**
     * Get widget icon.
     *
     * Retrieve number box widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */

    public function get_icon() {
        return 'eicon-icon-box';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @return array Widget keywords.
     * @since 2.1.0
     * @access public
     *
     */
    public function get_keywords() {
        return ['number box', 'number'];
    }

    public function get_script_depends() {
        return ['makbet-elementor-number-box'];
    }

    /**
     * Register number box widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls() {
        $this->start_controls_section(
            'section_number',
            [
                'label' => __('Number Box', 'makbet'),
            ]
        );

        $this->add_control(
            'selected_number',
            [
                'label'   => __('Number', 'makbet'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            ]
        );

        $this->add_control(
            'title_text',
            [
                'label'       => __('Title & Description', 'makbet'),
                'type'        => Controls_Manager::TEXT,
                'dynamic'     => [
                    'active' => true,
                ],
                'default'     => __('This is the heading', 'makbet'),
                'placeholder' => __('Enter your title', 'makbet'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'description_text',
            [
                'label'       => '',
                'type'        => Controls_Manager::TEXTAREA,
                'dynamic'     => [
                    'active' => true,
                ],
                'default'     => __('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'makbet'),
                'placeholder' => __('Enter your description', 'makbet'),
                'rows'        => 10,
                'separator'   => 'none',
                'show_label'  => false,
            ]
        );

        $this->add_control(
            'link',
            [
                'label'       => __('Link', 'makbet'),
                'type'        => Controls_Manager::URL,
                'dynamic'     => [
                    'active' => true,
                ],
                'placeholder' => __('https://your-link.com', 'makbet'),
                'separator'   => 'before',
            ]
        );

        $this->add_control(
            'position',
            [
                'label'        => __('Number Position', 'makbet'),
                'type'         => Controls_Manager::CHOOSE,
                'default'      => 'top',
                'options'      => [
                    'left'  => [
                        'title' => __('Left', 'makbet'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'top'   => [
                        'title' => __('Top', 'makbet'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'right' => [
                        'title' => __('Right', 'makbet'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'prefix_class' => 'elementor-position-',
                'toggle'       => false,
                'conditions'   => [
                    'relation' => 'or',
                    'terms'    => [
                        [
                            'name'     => 'selected_number',
                            'operator' => '!=',
                            'value'    => '',
                        ],
                    ],
                ],
            ]
        );

        $this->add_control(
            'title_size',
            [
                'label'   => __('Title HTML Tag', 'makbet'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'h1'   => 'H1',
                    'h2'   => 'H2',
                    'h3'   => 'H3',
                    'h4'   => 'H4',
                    'h5'   => 'H5',
                    'h6'   => 'H6',
                    'div'  => 'div',
                    'span' => 'span',
                    'p'    => 'p',
                ],
                'default' => 'h3',
            ]
        );

        $this->end_controls_section();

        //Wrapper
        $this->start_controls_section(
            'number_box_wrapper',
            [
                'label' => esc_html__('Wrapper', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs('tabs_wrapper_style');

        $this->start_controls_tab(
            'tab_wrapper_normal',
            [
                'label' => esc_html__('Normal', 'makbet'),
            ]
        );

        $this->add_control(
            'background_wrapper',
            [
                'label'     => esc_html__('Background', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} ' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_wrapper_hover',
            [
                'label' => esc_html__('Hover', 'makbet'),
            ]
        );

        $this->add_control(
            'background_wrapper_hover',
            [
                'label'     => esc_html__('Background', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}}:hover ' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'border_wrapper_hover',
            [
                'label'     => esc_html__('Border Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}}:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();


        $this->add_group_control(

            Group_Control_Border::get_type(),
            [
                'name'        => 'border_wrapper',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} ',
                'separator'   => 'before',
            ]
        );

        $this->add_control(
            'wrapper_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} ' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'wrapper_padding',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}}' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'wrapper_margin',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}}' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Number

        $this->start_controls_section(
            'section_style_number',
            [
                'label'      => __('Number', 'makbet'),
                'tab'        => Controls_Manager::TAB_STYLE,
                'conditions' => [
                    'relation' => 'or',
                    'terms'    => [
                        [
                            'name'     => 'selected_number',
                            'operator' => '!=',
                            'value'    => '',
                        ],
                    ],
                ],
            ]
        );



        $this->add_responsive_control(
            'number_space',
            [
                'label'     => __('Spacing', 'makbet'),
                'type'      => Controls_Manager::SLIDER,
                'default'   => [
                    'size' => 15,
                ],
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}}.elementor-position-right .elementor-number-box-number' => 'margin-left: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}}.elementor-position-left .elementor-number-box-number'  => 'margin-right: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}}.elementor-position-top .elementor-number-box-number'   => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '(mobile){{WRAPPER}} .elementor-number-box-number'                  => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'number_stroke_width',
            [
                'label'     => __('Stroke Width', 'makbet'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 10,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-number .svg-gradient-text text' => 'stroke-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'size_typography',
                'selector' => '{{WRAPPER}} .elementor-number .svg-gradient-text text',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_content',
            [
                'label' => __('Content', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'text_align',
            [
                'label'     => __('Alignment', 'makbet'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'left'    => [
                        'title' => __('Left', 'makbet'),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center'  => [
                        'title' => __('Center', 'makbet'),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'   => [
                        'title' => __('Right', 'makbet'),
                        'icon'  => 'eicon-text-align-right',
                    ],
                    'justify' => [
                        'title' => __('Justified', 'makbet'),
                        'icon'  => 'eicon-text-align-justify',
                    ],
                ],
                'default'   => 'center',
                'selectors' => [
                    '{{WRAPPER}} .elementor-number-box-wrapper' => 'text-align: {{VALUE}};',
                    '{{WRAPPER}} .elementor-number-box-wrapper .elementor-number-box-number .elementor-number' => 'text-align: -webkit-{{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'content_vertical_alignment',
            [
                'label'        => __('Vertical Alignment', 'makbet'),
                'type'         => Controls_Manager::SELECT,
                'options'      => [
                    'top'    => __('Top', 'makbet'),
                    'middle' => __('Middle', 'makbet'),
                    'bottom' => __('Bottom', 'makbet'),
                ],
                'default'      => 'top',
                'prefix_class' => 'elementor-vertical-align-',
            ]
        );

        $this->add_control(
            'heading_title',
            [
                'label'     => __('Title', 'makbet'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'title_bottom_space',
            [
                'label'     => __('Spacing', 'makbet'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-number-box-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => __('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-number-box-content .elementor-number-box-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_color_hover',
            [
                'label'     => __('Color Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}}:hover .elementor-number-box-content .elementor-number-box-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .elementor-number-box-content .elementor-number-box-title',
            ]
        );

        $this->add_control(
            'heading_description',
            [
                'label'     => __('Description', 'makbet'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label'     => __('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-number-box-content .elementor-number-box-description' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'description_color_hover',
            [
                'label'     => __('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}}:hover .elementor-number-box-content .elementor-number-box-description' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'description_typography',
                'selector' => '{{WRAPPER}} .elementor-number-box-content .elementor-number-box-description',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render icon box widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        $this->add_render_attribute('number', 'class', ['elementor-number']);

        $icon_tag = 'span';


        if (!empty($settings['link']['url'])) {
            $icon_tag = 'a';

            $this->add_link_attributes('link', $settings['link']);
        }


        $link_attributes = $this->get_render_attribute_string('link');

        $this->add_render_attribute('description_text', 'class', 'elementor-number-box-description');

        $this->add_inline_editing_attributes('title_text', 'none');
        $this->add_inline_editing_attributes('description_text');

        ?>
        <div class="elementor-number-box-wrapper">

            <?php if (!empty($settings['selected_number'])): ?>
                <div class="elementor-number-box-number">
                    <div class="elementor-number">
                        <?php
                        $randomNumber = rand(1000, 9999);
                        $stroke = 'gradient-stroke-'.$randomNumber;
                        ?>

                        <style>
                            .elementor-widget-number-box:hover .elementor-number-box-wrapper svg text {
                                fill: url(#<?php echo esc_attr($stroke); ?>);
                            }
                        </style>

                        <svg xmlns="http://www.w3.org/2000/svg" class="svg-gradient-text">
                            <defs>
                                <linearGradient id="<?php echo esc_attr($stroke); ?>" x1="0%" y1="0%" x2="100%" y2="0%">
                                    <stop offset="0%" stop-color="#FF5B46"/>
                                    <stop offset="100%" stop-color="#3A3CF2"/>
                                </linearGradient>
                            </defs>

                            <text x="0" y="0" stroke="url(#<?php echo esc_attr($stroke); ?>">
                                <?php echo esc_html($settings['selected_number']); ?>
                            </text>
                        </svg>
                    </div>
                </div>
            <?php endif; ?>
            <div class="elementor-number-box-content">
                <<?php Utils::print_unescaped_internal_string($settings['title_size']) ?> class="elementor-number-box-title">
                    <<?php echo wp_kses_post(implode(' ', [$icon_tag, $link_attributes])); ?><?php $this->print_render_attribute_string('title_text'); ?>>
                        <?php Utils::print_unescaped_internal_string($settings['title_text']) ?>
                    </<?php Utils::print_unescaped_internal_string($icon_tag); ?>>
                </<?php Utils::print_unescaped_internal_string($settings['title_size'] )?>>

                <?php if (!Utils::is_empty($settings['description_text'])) : ?>
                    <p <?php $this->print_render_attribute_string('description_text'); ?>><?php Utils::print_unescaped_internal_string($settings['description_text'] ) ?></p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

}

$widgets_manager->register(new OSF_Widget_Number_Box());
