<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Makbet\Elementor\Makbet_Base_Widgets;
use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;

class Makbet_Elementor_Service extends Makbet_Base_Widgets {

    public function get_name() {
        return 'makbet-service';
    }

    public function get_title() {
        return esc_html__('Makbet Service', 'makbet');
    }

    /**
     * Get widget icon.
     *
     * Retrieve testimonial widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-archive';
    }

    public function get_categories() {
        return array('makbet-addons');
    }

    public function get_script_depends() {
        return ['makbet-elementor-service'];
    }

    public function get_style_depends() {
        return ['e-swiper'];
    }

    protected function register_controls() {
        $this->start_controls_section(
            'section_query',
            [
                'label' => esc_html__('Query', 'makbet'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'posts_per_page',
            [
                'label'   => esc_html__('Posts Per Page', 'makbet'),
                'type'    => Controls_Manager::NUMBER,
                'default' => 8,
            ]
        );


        $this->add_control(
            'advanced',
            [
                'label' => esc_html__('Advanced', 'makbet'),
                'type'  => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label'   => esc_html__('Order By', 'makbet'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'post_date',
                'options' => [
                    'post_date'  => esc_html__('Date', 'makbet'),
                    'post_title' => esc_html__('Title', 'makbet'),
                    'menu_order' => esc_html__('Menu Order', 'makbet'),
                    'rand'       => esc_html__('Random', 'makbet'),
                ],
            ]
        );

        $this->add_control(
            'order',
            [
                'label'   => esc_html__('Order', 'makbet'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'desc',
                'options' => [
                    'asc'  => esc_html__('ASC', 'makbet'),
                    'desc' => esc_html__('DESC', 'makbet'),
                ],
            ]
        );

        $this->add_control(
            'categories',
            [
                'label'       => esc_html__('Categories', 'makbet'),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->get_post_categories(),
                'label_block' => true,
                'multiple'    => true,
            ]
        );

        $this->add_control(
            'cat_operator',
            [
                'label'     => esc_html__('Category Operator', 'makbet'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'IN',
                'options'   => [
                    'AND'    => esc_html__('AND', 'makbet'),
                    'IN'     => esc_html__('IN', 'makbet'),
                    'NOT IN' => esc_html__('NOT IN', 'makbet'),
                ],
                'condition' => [
                    'categories!' => ''
                ],
            ]
        );

        $this->add_control(
            'posts',
            [
                'label'       => esc_html__('Posts', 'makbet'),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->get_posts(),
                'label_block' => true,
                'multiple'    => true,
            ]
        );

        $this->add_control(
            'layout',
            [
                'label' => esc_html__('Layout', 'makbet'),
                'type'  => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'service_style',
            [
                'label'   => esc_html__('Style', 'makbet'),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    '1' => esc_html__('Style 1', 'makbet'),
                    '2' => esc_html__('Style 2', 'makbet'),
                    '3' => esc_html__('Style 3', 'makbet'),
                    '4' => esc_html__('Style 4', 'makbet'),
                ],
                'default' => '1'
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name'      => 'service_image',
                //                'default'   => 'makbet-service',
                'default'   => 'full',
                'separator' => 'none',
                'condition' => [
                    'service_style' => ['4', '3'],
                ],
            ]
        );

        $this->end_controls_section();

        // Wrapper
        $this->start_controls_section(
            'section_style_item',
            [
                'label' => esc_html__('Item', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name'      => 'background_color',
                'types'     => ['classic', 'gradient'],
                'selector'  => '{{WRAPPER}} .our-service',
                'condition' => [
                    'service_style' => ['1', '2'],
                ],
            ]
        );

        $this->add_responsive_control(
            'service_padding',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .our-service'                         => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .makbet-service-style-3'              => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .service-style-4 .service-title-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'service_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .our-service' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'service_style' => ['1', '2'],
                ],
            ]
        );

        $this->add_responsive_control(
            'service_spacing',
            [
                'label'      => esc_html__('Spacing', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem', '%', 'custom'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 200,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .our-service'                      => 'gap: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .service-style-4 .service-content' => 'gap: {{SIZE}}{{UNIT}}',
                ],
                'condition'  => [
                    'service_style' => ['1', '4'],
                ],
            ]
        );

        $this->end_controls_section();

        // Title
        $this->start_controls_section(
            'section_style_title',
            [
                'label' => esc_html__('Title', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'makbet'),
                'type'  => Controls_Manager::HEADING,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .service-content .entry-title, 
                                {{WRAPPER}} .service-style-2 .elementor-button .elementor-button-text',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service-content .entry-title'                             => 'color: {{VALUE}};',
                    '{{WRAPPER}} .service-style-2 .elementor-button .elementor-button-text' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .service-style-3 .entry-title'                             => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_color_hover',
            [
                'label'     => esc_html__('Color Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service-content .entry-title a:hover'                           => 'color: {{VALUE}};',
                    '{{WRAPPER}} .service-style-2 .elementor-button .elementor-button-text:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .service-style-3 .entry-title:hover'                             => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'service_style!' => ['4'],
                ],
            ]
        );

        $this->add_responsive_control(
            'style_title_margin',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-content .entry-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'service_style!' => ['2'],
                ],
            ]
        );

        $this->end_controls_section();

        // Description
        $this->start_controls_section(
            'section_style_description',
            [
                'label'     => esc_html__('Description', 'makbet'),
                'tab'       => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'service_style!' => ['2'],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'description_typography',
                'selector' => '{{WRAPPER}} .service-content .excerpt-content',
            ]
        );

        $this->add_responsive_control(
            'style_description_margin',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-content .excerpt-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Image
        $this->start_controls_section(
            'section_style_image',
            [
                'label'     => esc_html__('Image', 'makbet'),
                'tab'       => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'service_style!' => ['1', '2'],
                ],
            ]
        );

        $this->add_responsive_control(
            'style_image_height',
            [
                'label'      => esc_html__('Height', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem', '%', 'custom'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .service-image' => 'max-height: {{SIZE}}{{UNIT}}',
                ],
                'condition'  => [
                    'service_style' => ['4'],
                ],
            ]
        );

        $this->add_responsive_control(
            'style_image_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-image'                      => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .service-style-4 .service-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Icon

        $this->start_controls_section(
            'style_icon',
            [
                'label'     => esc_html__('Icon', 'makbet'),
                'tab'       => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'service_style' => ['1', '4'],
                ],
            ]
        );

        $this->add_responsive_control(
            'style_icon_size',
            [
                'label'      => esc_html__('Size', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem', '%', 'custom'],
                'range'      => [
                    'px' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .our-service.service-style-1' => '--size-icon: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .service-style-4'             => '--size-icon: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'style_icon_size_bg',
            [
                'label'      => esc_html__('Size Background', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem', '%', 'custom'],
                'range'      => [
                    'px' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .our-service.service-style-1' => '--size-bg-icon: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .service-style-4'             => '--size-bg-icon: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->start_controls_tabs(
            'style_tabs_service_color'
        );

        $this->start_controls_tab(
            'style_normal_tab_service_color',
            [
                'label'     => esc_html__('Normal', 'makbet'),
                'condition' => [
                    'service_style' => '1',
                ],
            ]
        );

        $this->add_control(
            'bg_color',
            [
                'label'     => esc_html__('Background Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-service .entry-icon' => 'background: {{VALUE}};',
                    '{{WRAPPER}} .timeline-dot'            => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name'           => 'icon_color',
                'types'          => ['gradient'],
                'selector'       => '{{WRAPPER}} .icon-mask',
                'fields_options' => [
                    'background' => [
                        'label'              => __('Icon Gradient', 'makbet'),
                        'frontend_available' => true,
                    ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'style_hover_tab_service_color',
            [
                'label'     => esc_html__('Hover', 'makbet'),
                'condition' => [
                    'service_style' => '1',
                ],
            ]
        );

        $this->add_control(
            'bg_color_hover',
            [
                'label'     => esc_html__('Background Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-service:hover .entry-icon' => 'background: {{VALUE}};',
                ],
                'condition' => [
                    'service_style' => '1',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name'           => 'icon_color_hover',
                'types'          => ['gradient'],
                'selector'       => '{{WRAPPER}} .our-service:hover .icon-mask',
                'fields_options' => [
                    'background' => [
                        'label'              => __('Icon Gradient Hover', 'makbet'),
                        'frontend_available' => true,
                    ],
                ],
                'condition'      => [
                    'service_style' => '1',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();

        $this->get_controls_column([
            'service_style' => ['1', '2'],
        ]);
        $this->get_control_carousel([
            'service_style' => ['1', '2'],
        ]);
    }

    protected function get_post_categories() {
        $categories = get_terms(array(
                'taxonomy'   => 'pavo_service_cat',
                'hide_empty' => false,
            )
        );
        $results    = array();
        if (!is_wp_error($categories)) {
            foreach ($categories as $category) {
                $results[$category->slug] = $category->name;
            }
        }
        return $results;
    }

    protected function get_posts() {
        $argc = [
            'numberposts' => -1,
            'post_type'   => 'pavo_service',
        ];

        $posts = get_posts($argc);

        $results = array();

        if (!empty($posts)) {
            foreach ($posts as $post) {
                $results[$post->ID] = html_entity_decode(get_the_title($post->ID), ENT_QUOTES, 'UTF-8');
            }
        }

        return $results;
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $query    = $this->query_posts();

        if (!$query->found_posts) {
            return;
        }

        $style = $settings['service_style'];

        if (in_array($style, ['3', '4'])) {
            $items = [];
            $i     = 0;

            while ($query->have_posts()) {
                $query->the_post();

                $items[] = [
                    'image'   => Group_Control_Image_Size::get_attachment_image_src(
                        get_post_thumbnail_id(),
                        'service_image',
                        $settings
                    ),
                    'title'   => get_the_title(),
                    'link'    => get_permalink(),
                    'excerpt' => get_the_excerpt(),
                    'icon'    => get_post_meta(get_the_ID(), 'service_icon', true),
                ];

                $i++;
            }
            wp_reset_postdata();

            if ($style === '3') { ?>
                <div class="service-wrapper service-style-3">
                    <div class="service-list">
                        <?php foreach ($items as $index => $item): ?>
                            <div class="service makbet-service-style-3"
                                 data-link="<?php echo esc_url($item['link']); ?>">
                                <div class="service-inner">
                                    <div class="service-content">
                                        <span class="service-number">
                                            <?php
                                            $randomNumber = rand(1000, 9999);
                                            $stroke = 'gradient-stroke-'.$randomNumber;
                                            ?>

                                            <style>
                                                 .service-wrapper .makbet-service-style-3.active svg text {
                                                     fill: url(#<?php echo esc_attr($stroke); ?>);
                                                 }
                                            </style>

                                            <svg xmlns="http://www.w3.org/2000/svg" class="svg-gradient-text">
                                                <defs>
                                                    <linearGradient id="<?php echo esc_attr($stroke); ?>" x1="0%" y1="0%" x2="100%" y2="0%">
                                                        <stop offset="0%" stop-color="#FF5B46"/>
                                                        <stop offset="100%" stop-color="#3A3CF2"/>
                                                    </linearGradient>
                                                </defs>

                                                <text x="0" y="0" style="stroke: url(#<?php echo esc_attr($stroke); ?>);">
                                                    <?php echo str_pad($index + 1, 2, '0', STR_PAD_LEFT); ?>
                                                </text>
                                            </svg>
                                        </span>
                                        <div class="entry-main">
                                            <a href="javascript:void(0)">
                                                <h4 class="entry-title">
                                                    <?php echo esc_html($item['title']); ?>
                                                </h4>
                                            </a>
                                            <div class="excerpt-content">
                                                <?php echo esc_html($item['excerpt']); ?>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="service-icon">
                                        <i class="makbet-icon- makbet-icon-arrow-right-normal"></i>
                                    </div>
                                    <div class="service-image">
                                        <img src="<?php echo esc_url($item['image']); ?>"
                                             alt="<?php echo esc_attr($item['title']); ?>"
                                             loading="lazy">
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php return;
            }

            if ($style === '4') {
                $main_service = $items[0] ?? ['image' => '', 'title' => '', 'link' => '', 'excerpt' => ''];
                ?>
                <?php if (!empty($items)) : ?>
                    <section class="scroll-lock-section service-style-4">
                        <div class="service-content">
                            <div class="service-list-wrapper">
                                <div class="service-list">
                                    <?php foreach ($items as $item): ?>
                                        <div class="service-title-item" data-image="<?php echo esc_url($item['image']); ?>">
                                            <div class="timeline-dot">
                                                <div class="icon-mask" style="--icon-url: url('<?php echo esc_url($item['icon']); ?>');"></div>
                                            </div>

                                            <div class="timeline-line-wrapper">
                                                <div class="timeline-line"></div>
                                                <div class="timeline-line-fill"></div>
                                            </div>

                                            <div class="service-content-main">
                                                <h3 class="entry-title"><?php echo esc_html($item['title']); ?></h3>
                                                <?php if (!empty($item['excerpt'])) : ?>
                                                    <div class="excerpt-content"><?php echo esc_html($item['excerpt']); ?></div>
                                                <?php endif; ?>
                                                <div class="elementor-button-link button-service">
                                                    <a class="elementor-button has-interacted" href="<?php echo esc_url($item['link']); ?>">
                                                        <span class="elementor-button-content-wrapper">
                                                            <span class="elementor-button-text"><?php echo esc_html__( 'Learn more', 'makbet' ); ?></span>
                                                            <span class="elementor-button-icon">
                                                                <i aria-hidden="true" class="makbet-icon- makbet-icon-arrow-right-normal"></i>
                                                            </span>
                                                        </span>
                                                    </a>
                                                </div>

                                                <div class="mobile-preview">
                                                    <img src="<?php echo esc_url($item['image']); ?>" alt="<?php echo esc_attr($item['title']); ?>"/>
                                                </div>
                                            </div>

                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <div class="service-image">
                                <?php foreach ($items as $item): ?>
                                    <img src="<?php echo esc_url($item['image']); ?>" alt="<?php echo esc_attr($item['title']); ?>"/>
                                <?php endforeach; ?>
                            </div>

                        </div>
                    </section>
                <?php endif;
                return;
            }

        }

        $this->add_render_attribute('wrapper', 'class', ['elementor-service-wrapper']);
        $this->add_render_attribute('container', 'data-count', $query->post_count);

        $this->add_render_attribute('item', 'class', 'elementor-service-item');

        $this->get_data_elementor_columns();
        ?>
        <div <?php $this->print_render_attribute_string('wrapper'); ?>>
            <div <?php $this->print_render_attribute_string('container'); ?>>
                <div <?php $this->print_render_attribute_string('inner'); ?>>
                    <?php
                    $query->rewind_posts();
                    while ($query->have_posts()) {
                        $query->the_post(); ?>
                        <div <?php $this->print_render_attribute_string('item'); ?>>
                            <?php include get_theme_file_path('template-parts/service/content-style-' . $style . '.php'); ?>
                        </div>
                    <?php } ?>
                </div>

                <?php
                $this->get_swiper_navigation($query->post_count);
                ?>
            </div>
        </div>
        <?php
        wp_reset_postdata();
    }

    public function query_posts() {
        $query_args = $this->get_query_args($this->get_settings());
        return new WP_Query($query_args);
    }

    public static function get_query_args($settings) {
        $query_args = [
            'post_type'           => 'pavo_service',
            'orderby'             => $settings['orderby'],
            'order'               => $settings['order'],
            'ignore_sticky_posts' => 1,
            'post_status'         => 'publish',
        ];

        if (!empty($settings['categories'])) {
            $categories = array();
            foreach ($settings['categories'] as $category) {
                $cat = get_term_by('slug', $category, 'pavo_service_cat');
                if (!is_wp_error($cat) && is_object($cat)) {
                    $categories[] = $cat->term_id;
                }
            }

            $cat_operator = isset($settings['cat_operator']) ? $settings['cat_operator'] : 'IN';

            $query_args['tax_query'] = [
                'relation' => 'AND',
                [
                    'taxonomy' => 'pavo_service_cat',
                    'field'    => 'term_id',
                    'terms'    => $categories,
                    'operator' => $cat_operator,
                ],
            ];

        } elseif (!empty($settings['posts'])) {
            $query_args['post__in'] = $settings['posts'];
        }

        $query_args['posts_per_page'] = $settings['posts_per_page'];

        if (is_front_page()) {
            $query_args['paged'] = (get_query_var('page')) ? get_query_var('page') : 1;
        } else {
            $query_args['paged'] = (get_query_var('paged')) ? get_query_var('paged') : 1;
        }

        return $query_args;
    }

}

$widgets_manager->register(new Makbet_Elementor_Service());