<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Group_Control_Box_Shadow;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Border;
use Elementor\Repeater;
use Makbet\Elementor\Makbet_Base_Widgets;
use Elementor\Utils;
use Elementor\Icons_Manager;

class Makbet_Elementor_Team_Box extends Makbet_Base_Widgets {

    /**
     * Get widget name.
     *
     * Retrieve testimonial widget name.
     *
     * @return string Widget name.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'makbet-team-box';
    }

    /**
     * Get widget title.
     *
     * Retrieve testimonial widget title.
     *
     * @return string Widget title.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_title() {
        return esc_html__('Makbet Team Box', 'makbet');
    }

    /**
     * Get widget icon.
     *
     * Retrieve testimonial widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-person';
    }

    public function get_script_depends() {
        return ['makbet-elementor-team-box'];
    }

    public function get_categories() {
        return array('makbet-addons');
    }

    public function get_style_depends() {
        return ['elementor-icons-fa-solid', 'elementor-icons-fa-brands', 'elementor-icons-fa-regular'];
    }

    /**
     * Register testimonial widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function register_controls() {

        $this->start_controls_section(
            'section_team',
            [
                'label' => esc_html__('Team', 'makbet'),
            ]
        );

        // Nested Repeater: Social Icons
        $social_repeater = new Repeater();

        $social_repeater->add_control(
            'social_title',
            [
                'label'   => esc_html__('Title', 'makbet'),
                'type'    => Controls_Manager::TEXT,
                'default' => 'Facebook',
            ]
        );

        $social_repeater->add_control(
            'social_icon',
            [
                'label'            => esc_html__('Icon', 'makbet'),
                'type'             => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default'          => [
                    'value'   => 'fab fa-facebook-f',
                    'library' => 'fa-brands',
                ],
            ]
        );

        $social_repeater->add_control(
            'social_link',
            [
                'label'       => esc_html__('Link', 'makbet'),
                'type'        => Controls_Manager::URL,
                'placeholder' => esc_html__('https://your-social.com', 'makbet'),
            ]
        );

        // Team Repeater
        $repeater = new Repeater();

        $repeater->add_control(
            'teambox_image',
            [
                'label'   => esc_html__('Image', 'makbet'),
                'type'    => Controls_Manager::MEDIA,
                'default' => ['url' => \Elementor\Utils::get_placeholder_image_src()],
            ]
        );

        $repeater->add_control(
            'name',
            [
                'label'   => esc_html__('Name', 'makbet'),
                'type'    => Controls_Manager::TEXT,
                'default' => 'Morgan Maxwell',
            ]
        );

        $repeater->add_control(
            'job',
            [
                'label'   => esc_html__('Job', 'makbet'),
                'type'    => Controls_Manager::TEXT,
                'default' => 'CEO & Founder',
            ]
        );

        $repeater->add_control(
            'link',
            [
                'label'       => esc_html__('Link to', 'makbet'),
                'type'        => Controls_Manager::URL,
                'dynamic'     => ['active' => true],
                'placeholder' => esc_html__('https://your-link.com', 'makbet'),
            ]
        );

        $repeater->add_control(
            'socials',
            [
                'label'        => esc_html__('Social Links', 'makbet'),
                'type'         => Controls_Manager::REPEATER,
                'fields'       => $social_repeater->get_controls(),
                'title_field'  => '{{{ social_title }}}',
                'max_items'    => 5,
                'item_actions' => [
                    'add' => false,
                ],
                'default'      => [
                    [
                        'social_title' => esc_html__('Facebook', 'makbet'),
                    ],
                ],
            ]
        );

        $this->add_control(
            'teambox',
            [
                'label'         => esc_html__('Team Members', 'makbet'),
                'type'          => Controls_Manager::REPEATER,
                'fields'        => $repeater->get_controls(),
                'title_field'   => '{{{ name }}}',
                'prevent_empty' => false
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name'      => 'teambox_image',
                'default'   => 'full',
                'separator' => 'none',
            ]
        );


        $this->add_control(
            'teambox_layout',
            [
                'label'        => esc_html__('Layout', 'makbet'),
                'type'         => Controls_Manager::SELECT,
                'default'      => 'default',
                'options'      => [
                    'default' => esc_html__('Default', 'makbet'),
                ],
                'prefix_class' => 'teambox-layout-style-',
            ]
        );

        $this->end_controls_section();


        // Wrapper.
        $this->start_controls_section(
            'section_style_team_wrapper',
            [
                'label' => esc_html__('Items', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'team_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .elementor-teambox-item',
                'separator'   => 'before',
            ]
        );

        $this->add_responsive_control(
            'team_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-teambox-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'wrapper_padding_inner',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-teambox-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'wrapper_margin',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-teambox-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();


        // Image.
        $this->start_controls_section(
            'section_style_team_image',
            [
                'label' => esc_html__('Image', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'teambox_image_width',
            [
                'label'      => esc_html__('Width', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range'      => [
                    'px' => [
                        'min' => 1,
                        'max' => 900,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-teambox-item .team-image img' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'teambox_image_height',
            [
                'label'      => esc_html__('Height', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range'      => [
                    'px' => [
                        'min' => 1,
                        'max' => 900,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-teambox-item .team-image img' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .team-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Content
        $this->start_controls_section(
            'section_style_team_main',
            [
                'label' => esc_html__('Content', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'main_alignment',
            [
                'label'       => esc_html__('Alignment Content', 'makbet'),
                'type'        => Controls_Manager::CHOOSE,
                'options'     => [
                    'left'   => [
                        'title' => esc_html__('Left', 'makbet'),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'makbet'),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'  => [
                        'title' => esc_html__('Right', 'makbet'),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'label_block' => false,
                'selectors'   => [
                    '{{WRAPPER}} .team-main'         => 'text-align: {{VALUE}};',
                    '{{WRAPPER}} .team-icon-socials' => 'justify-content: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'main_padding',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .team-caption' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Name.
        $this->start_controls_section(
            'section_style_team_name',
            [
                'label' => esc_html__('Name', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'name_text_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-teambox-item .team-name a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .elementor-teambox-item .team-name'   => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'name_text_color_hover',
            [
                'label'     => esc_html__('Color Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-teambox-item .team-name a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .elementor-teambox-item .team-name:hover'   => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'name_typography',
                'selector' => '{{WRAPPER}} .elementor-teambox-item .team-name',
            ]
        );

        $this->add_responsive_control(
            'name_padding',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-teambox-item .team-name' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Job.
        $this->start_controls_section(
            'section_style_team_job',
            [
                'label' => esc_html__('Job', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'job_text_color',
            [
                'label'     => esc_html__('Text Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-teambox-item .team-job' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'job_typography',
                'selector' => '{{WRAPPER}} .elementor-teambox-item .team-job',
            ]
        );

        $this->add_responsive_control(
            'job_padding',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-teambox-item .team-job' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Icon Social.

        $this->start_controls_section(
            'section_style_icon_social',
            [
                'label' => esc_html__('Icon Social', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'teambox_icon_spacing',
            [
                'label'      => esc_html__('Spacing', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range'      => [
                    'px' => [
                        'min' => 1,
                        'max' => 900,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-teambox-item .team-icon-socials .team-socials-wrapper' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_nav_menu_style');

        $this->start_controls_tab(
            'tab_icon_normal',
            [
                'label' => esc_html__('Normal', 'makbet'),
            ]
        );

        $this->add_control(
            'color_icon_social',
            [
                'label'   => esc_html__('Color', 'makbet'),
                'type'    => Controls_Manager::COLOR,
                'default' => '',

                'selectors' => [
                    '{{WRAPPER}} .elementor-teambox-item .team-icon-socials .social a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_icon_hover',
            [
                'label' => esc_html__('Normal', 'makbet'),
            ]
        );

        $this->add_control(
            'color_icon_social_hover',
            [
                'label'     => esc_html__('Color Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-teambox-item .team-icon-socials .social a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();
        // Carousel column
        $this->get_controls_column();
        // Carousel options
        $this->get_control_carousel();
    }

    /**
     * Render testimonial widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        if (!empty($settings['teambox']) && is_array($settings['teambox'])) {
            $this->get_data_elementor_columns();
            $this->add_render_attribute('wrapper', 'class', 'elementor-teambox-item-wrapper');
            $this->add_render_attribute('container', 'data-count', count($settings['teambox']));
            // Item
            $this->add_render_attribute('item', 'class', 'elementor-teambox-item');
            $this->add_render_attribute('details', 'class', 'details');
            ?>

            <div <?php $this->print_render_attribute_string('wrapper'); // WPCS: XSS ok. ?>>
                <div <?php $this->print_render_attribute_string('container'); ?>>
                    <div <?php $this->print_render_attribute_string('inner'); // WPCS: XSS ok. ?>>
                        <?php foreach ($settings['teambox'] as $teambox): ?>
                            <div <?php $this->print_render_attribute_string('item'); // WPCS: XSS ok. ?>>

                                <div class="team-image">
                                    <?php $this->render_image($settings, $teambox); ?>

                                    <div class="team-icon-socials">
                                        <div class="team-socials-share">
                                            <i class="makbet-icon-share"></i>
                                        </div>
                                        <div class="team-socials-wrapper">
                                            <?php if (!empty($teambox['socials']) && is_array($teambox['socials'])) : ?>
                                                <?php foreach ($teambox['socials'] as $social) : ?>
                                                    <span class="social">
                                                        <a href="<?php echo esc_url($social['social_link']['url']); ?>"
                                                           title="<?php echo esc_html($social['social_title']); ?>"
                                                           target="<?php echo esc_attr($social['social_link']['is_external'] ? '_blank' : '_self'); ?>">
                                                            <?php Icons_Manager::render_icon($social['social_icon'], ['aria-hidden' => 'true']); ?>
                                                        </a>
                                                    </span>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>

                                <div class="team-caption">
                                    <div class="team-main">
                                        <div class="team-name">
                                            <?php
                                            $teambox_name_html = $teambox['name'];
                                            if (!empty($teambox['link']['url'])) :
                                                $teambox_name_html = '<a href="' . esc_url($teambox['link']['url']) . '">' . esc_html($teambox_name_html) . '</a>';
                                            endif;
                                            printf($teambox_name_html);
                                            ?>
                                        </div>

                                        <div class="team-job"><?php echo esc_html($teambox['job']); ?></div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                        <?php wp_reset_postdata(); ?>
                    </div>
                    <?php $this->get_swiper_navigation(count($settings['teambox'])); ?>
                </div>
            </div>

            <?php
        }

    }

    private function render_image($settings, $teambox) {
        if (!empty($teambox['teambox_image']['url'])) :
            $teambox['teambox_image_size']             = $settings['teambox_image_size'];
            $teambox['teambox_image_custom_dimension'] = $settings['teambox_image_custom_dimension'];
            echo Group_Control_Image_Size::get_attachment_image_html($teambox, 'teambox_image');
        endif;
    }

}

$widgets_manager->register(new Makbet_Elementor_Team_Box());
